#ifndef __DDP_H
#define __DDP_H
/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: AP_Ddp2.h 4490 2020-08-13 13:32:21Z Robert $:

  Description:
    Device data provider

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2018-12-19  Added structures for Ddp_SetData()
    2017-06-01  File created

**************************************************************************************/

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#define DDP_DEFAULT_NAME_SIZE                   (16)

/*************************************************************************************/
/*! DDP commands                                                                     */
/*************************************************************************************/
#define DDP_DATA_TYPE_BASE_DEVICE_DATA          (0)
#define DDP_DATA_TYPE_MAC_ADDRESS               (1)
#define DDP_DATA_TYPE_HARDWARE_INFO             (2)
#define DDP_DATA_TYPE_USB_IDENTIFICATION        (3)
#define DDP_DATA_TYPE_OEM_DEVICE_DATA           (4)
#define DDP_DATA_TYPE_LIBSTORAGE_DATA           (5)
#define DDP_DATA_TYPE_LICENSE_FLAGS             (6)
#define DDP_DATA_TYPE_SDRAM                     (7)
#define DDP_DATA_TYPE_STATE                     (8)
#define DDP_DATA_TYPE_ANY                       (~0)


/*************************************************************************************/
/*! Initial state parameter for Ddp_Init()                                           */
/*************************************************************************************/
typedef enum DDP_STATE_Etag {
  DDP_STATE_PASSIVE,
  DDP_STATE_ACTIVE
} DDP_STATE_E;


/*************************************************************************************/
/*! Input / Output for base device data                                              */
/*************************************************************************************/
typedef struct DDP_BASE_DEVICE_DATA_OUT_Ttag
{
  uint16_t  usManufacturer;
  uint16_t  usDeviceClass;
  uint32_t  ulDeviceNumber;
  uint32_t  ulSerialNumber;
  uint8_t   bHwCompatibility;
  uint8_t   bHwRevision;
  uint16_t  usProductionDate;
} DDP_BASE_DEVICE_DATA_OUT_T;

typedef DDP_BASE_DEVICE_DATA_OUT_T DDP_BASE_DEVICE_DATA_SET_IN_T;

/*************************************************************************************/
/*! Input / Output for MAC address                                                   */
/*************************************************************************************/
#define DDP_MAC_ADDRESS_TYPE_COM 0
#define DDP_MAC_ADDRESS_TYPE_APP 1

typedef struct DDP_MAC_ADDRESS_IN_Ttag
{
  uint32_t  ulPort;
  uint32_t  ulType;
} DDP_MAC_ADDRESS_IN_T;

typedef struct DDP_MAC_ADDRESS_OUT_Ttag
{
  uint8_t   abMacAddress[6];
} DDP_MAC_ADDRESS_OUT_T;

typedef struct DDP_MAC_ADDRESS_SET_IN_Ttag
{
  uint32_t  ulPort;
  uint32_t  ulType;
  uint8_t   abMacAddress[6];
} DDP_MAC_ADDRESS_SET_IN_T;

/*************************************************************************************/
/*! Input / Output for hardware information                                          */
/*************************************************************************************/
typedef struct DDP_HARDWARE_INFORMATION_OUT_Ttag
{
  uint16_t  ausHwOptionXc[4];
  uint16_t  ausHwOptionsXpic[4];
} DDP_HARDWARE_INFORMATION_OUT_T;

/*************************************************************************************/
/*! Input / Output for USB information                                               */
/*************************************************************************************/
#define DDP_USB_PRODUCT_NAME_SIZE   DDP_DEFAULT_NAME_SIZE
#define DDP_USB_VENDOR_NAME_SIZE    DDP_DEFAULT_NAME_SIZE

typedef struct DDP_USB_INFO_OUT_Ttag
{
  uint16_t  usUSBVendorID;
  uint16_t  usUSBProductID;
  uint8_t   abUSBVendorName[DDP_USB_VENDOR_NAME_SIZE];
  uint8_t   abUSBProductName[DDP_USB_PRODUCT_NAME_SIZE];
} DDP_USB_INFO_OUT_T;

typedef DDP_USB_INFO_OUT_T DDP_USB_INFO_SET_IN_T;

/*************************************************************************************/
/*! Input / Output for OEM information                                               */
/*************************************************************************************/
typedef struct DDP_OEM_IDENTIFICATION_OUT_Ttag
{
  uint32_t  ulOemDataOptionFlags;
  char      szSerialNumber[28];
  char      szOrderNumber[32];
  char      szHardwareRevision[16];
  char      szProductionDate[32];
  uint8_t   abReserved[12];
  uint8_t   abVendorData[112];
} DDP_OEM_IDENTIFICATION_OUT_T;

typedef DDP_OEM_IDENTIFICATION_OUT_T DDP_OEM_IDENTIFICATION_SET_IN_T;

/*************************************************************************************/
/*! Input / Output for storage library information                                   */
/*************************************************************************************/
#define DDP_LIBSTORAGE_TYPE_AREA 1
#define DDP_LIBSTORAGE_TYPE_CHIP 2

typedef struct DDP_LIBSTORAGE_IN_Ttag
{
  uint32_t  ulType;
  uint32_t  ulId;
} DDP_LIBSTORAGE_IN_T;

typedef struct DDP_LIBSTORAGE_AREA_OUT_Ttag
{
  uint32_t  ulContentType;
  uint32_t  ulAreaStart;
  uint32_t  ulAreaSize;
  uint32_t  ulChipNumber;
  char      szName[DDP_DEFAULT_NAME_SIZE];
  uint8_t   bAccessTyp;
  uint8_t   abReserved[3];
} DDP_LIBSTORAGE_AREA_OUT_T;

typedef struct DDP_LIBSTORAGE_CHIP_OUT_Ttag
{
  uint32_t  ulChipNumber;
  char      szFlashName[DDP_DEFAULT_NAME_SIZE];
  uint32_t  ulBlockSize;
  uint32_t  ulFlashSize;
  uint32_t  ulMaxEnduranceCycles;
} DDP_LIBSTORAGE_CHIP_OUT_T;

typedef struct DDP_LIBSTORAGE_OUT_Ttag
{
  DDP_LIBSTORAGE_AREA_OUT_T atArea[10];
  DDP_LIBSTORAGE_CHIP_OUT_T atChip[4];
} DDP_LIBSTORAGE_OUT_T;

/*************************************************************************************/
/*! Input / Output for license information                                           */
/*************************************************************************************/
typedef struct DDP_LICENSE_FLAGS_OUT_Ttag
{
  uint32_t  ulLicenseId;
  uint32_t  ulLicenseFlags1;
  uint32_t  ulLicenseFlags2;
} DDP_LICENSE_FLAGS_OUT_T;

/*************************************************************************************/
/*! Input / Output for SDRAM information                                             */
/*************************************************************************************/
#define DDP_SDRAM_TYPE_NONE   0
#define DDP_SDRAM_TYPE_HIFMEM 1
#define DDP_SDRAM_TYPE_EXTMEM 2

typedef struct DDP_SDRAM_OUT_Ttag
{
  uint32_t  ulStart;
  uint8_t   bSize;
  uint8_t   bType;
} DDP_SDRAM_OUT_T;

/*************************************************************************************/
/*! Input / Output DDP state information                                             */
/*************************************************************************************/
typedef struct DDP_STATE_DATA_OUT_Ttag
{
  DDP_STATE_E eState;
} DDP_STATE_DATA_OUT_T;

typedef DDP_STATE_DATA_OUT_T DDP_STATE_DATA_SET_IN_T;

/*************************************************************************************/
/*! Initialization parameter for Ddp_Ini()                                           */
/*************************************************************************************/
typedef struct DDP_INIT_DATA_Ttag
{
  struct
  {
    uint16_t usUSBVendorID;
    uint16_t usUSBProductID;
    char*    pabUSBVendorName;
    char*    pabUSBProductName;
  } tUsbData;

} DDP_INIT_DATA_T;

/*************************************************************************************/
/*! Function typedef for notification callback                                       */
/*************************************************************************************/
typedef void(*DDP_NOTIFY_PFN)(uint32_t ulDataType, void* pvUser);

/*************************************************************************************/
/*! DDP API functions                                                                */
/*************************************************************************************/
uint32_t Ddp_Init         (DDP_STATE_E eState, void* pvParam);
uint32_t Ddp_Deinit       (void);
uint32_t Ddp_GetData      (uint32_t ulDataType, uint32_t ulInSize,  void* pvIn,
                                                uint32_t ulOutSize, void* pvOut);
uint32_t Ddp_SetData      (uint32_t ulDataType, uint32_t ulInSize,  void* pvIn);

uint32_t Ddp_Subscribe    (DDP_NOTIFY_PFN pfnNotify, void* pvUser, void** phNotify);
uint32_t Ddp_Unsubscribe  (void* phNotify);

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#endif /* __DDP_H */
